function X = low_rank(y, U, N, options)

% INPUT
% y - measurement vector (time domain, dimensions M x 1)
% U - sampling scheme with increment numbers (M x 1)
% N - length of full signal

% PARAMETRES
% options.lambda - data agreement/sparsity [default lambda = 10]
% options.Q - number of columns in Hankel matrix [default Q = ceil(0.1*N)]
% options.beta [default beta = 1] 
% options.tau - "step" [default tau = beta = 1] 
% options.tol - tolerance of convergence [default tol = 1e-8]
% options.max_iter - limit of iterations [default max_iter = Inf, no limit]

% OUTPUT
% X (frequency domain, N x 1)

% POSSIBLE MODIFICATIONS: 
% interruption at low singular values: options.sv 

%%%%%%%%%%%% default parameters %%%%%%%%%%%%
if isfield(options, 'max_iter') 
    max_iter = options.max_iter;
else
    max_iter = Inf; % no upper limit to the number of iterations by default
end

if isfield(options, 'tol') 
    tol = options.tol;
else
    tol = 1e-5;
end

if isfield(options, 'beta') 
    beta = options.beta;
else
    beta = 1;
end

if isfield(options, 'tau') 
    tau = options.tau;
else
    tau = beta;
end


if isfield(options, 'Q') 
    Q = options.Q;
else
    Q = ceil(0.1*N);
end

if isfield(options, 'lambda') 
    lambda = options.lambda;
else
    lambda = 1e3;
end

%%%%%%%%%%%% modifications %%%%%%%%%%%%
if isfield(options, 'sv') % modification 2
    sv = options.sv;
end

%%%%%%%%%%%% initialization %%%%%%%%%%%%
maxsig = abs(max(abs(y)));
y = y/maxsig;

% form Hankel operator R: R*X = H (H - Hankel matrix):
row = 1:((N - Q + 1)*Q);
row_block = floor(row/(N - Q + 1) - 1e-5); % 1e-5 is introduced to "shift" division "backwards"
sr = ones((N - Q + 1)*Q, 1);
R = sparse(row, row - (N - Q + 1)*row_block + row_block, sr); % Hankel operator R

% form sampling matrix Um (M x N) from sampling vector U (M x 1):
su = ones(length(U), 1);
Um = sparse([1:length(U)], U, su, length(U), N);

X = Um'*y;
x_last = X;
delta_x = 2*tol;
Z = R*X;
D = ones(Q*(N-Q+1), 1);% lagrangian multiplier

%%%%%%%%%%%% main loop %%%%%%%%%%%%
a = lambda*Um'*y;
b = beta*R';
c  = (lambda*Um'*Um + beta*R'*R)^(-1); 

k = 0;
while delta_x >= tol && k < max_iter
%     disp(horzcat('iteration', ' ', int2str(k)))
    X = c*(a + b*(Z - D/beta)); 
    k = k + 1;
    A = R*X + D/beta;
    A = reshape(A, N - Q + 1, Q);
    [u, s, v] = svd(A, 'econ');
    if isfield(options, 'sv')
        if max(diag(s)) < sv/beta
            x = zeros(N, 1);
            break
            'singular value limit has been reached'
        end
    end
    Z = u*max(0, s - 1/beta)*v'; % soft thresholding on sihgular values 
    Z = reshape(Z, Q*(N - Q + 1), 1); 
    D = D + tau*(R*X - Z); 
    delta_x = norm(x_last - X)/norm(x_last);
    x_last = X;
end
X = X*maxsig;
end
